/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal.audio;

import android.content.Context;
import androidx.annotation.RequiresPermission;
import androidx.annotation.VisibleForTesting;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.Observable;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.impl.utils.futures.FutureCallback;
import androidx.camera.core.impl.utils.futures.Futures;
import androidx.camera.video.internal.BufferProvider;
import androidx.camera.video.internal.audio.AudioSettings;
import androidx.camera.video.internal.audio.AudioSourceAccessException;
import androidx.camera.video.internal.audio.AudioStream;
import androidx.camera.video.internal.audio.AudioStreamFactory;
import androidx.camera.video.internal.audio.AudioStreamImpl;
import androidx.camera.video.internal.audio.BufferedAudioStream;
import androidx.camera.video.internal.audio.SilentAudioStream;
import androidx.camera.video.internal.encoder.InputBuffer;
import androidx.concurrent.futures.CallbackToFutureAdapter;
import androidx.core.util.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import java.nio.ByteBuffer;
import java.nio.ShortBuffer;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

public final class AudioSource {
    private static final String TAG = "AudioSource";
    @VisibleForTesting
    static final long DEFAULT_START_RETRY_INTERVAL_MS = 3000L;
    final Executor mExecutor;
    final AtomicReference<Boolean> mNotifiedSilenceState = new AtomicReference<Object>(null);
    final AtomicBoolean mNotifiedSuspendState = new AtomicBoolean(false);
    final AudioStream mAudioStream;
    final SilentAudioStream mSilentAudioStream;
    private final long mStartRetryIntervalNs;
    @NonNull InternalState mState = InternalState.CONFIGURED;
    @NonNull BufferProvider.State mBufferProviderState = BufferProvider.State.INACTIVE;
    boolean mIsSendingAudio;
    @Nullable Executor mCallbackExecutor;
    @Nullable AudioSourceCallback mAudioSourceCallback;
    @Nullable BufferProvider<? extends InputBuffer> mBufferProvider;
    private @Nullable FutureCallback<InputBuffer> mAcquireBufferCallback;
    private // Could not load outer class - annotation placement on inner may be incorrect
     @Nullable Observable.Observer<BufferProvider.State> mStateObserver;
    boolean mInSilentStartState;
    private long mLatestFailedStartTimeNs;
    boolean mAudioStreamSilenced;
    boolean mMuted;
    private byte @Nullable [] mZeroBytes;
    double mAudioAmplitude;
    long mAmplitudeTimestamp = 0L;
    private final int mAudioFormat;
    @VisibleForTesting
    public final int mAudioSource;

    @RequiresPermission(value="android.permission.RECORD_AUDIO")
    public AudioSource(@NonNull AudioSettings settings, @NonNull Executor executor, @Nullable Context attributionContext) throws AudioSourceAccessException {
        this(settings, executor, attributionContext, AudioStreamImpl::new, 3000L);
    }

    @VisibleForTesting
    @RequiresPermission(value="android.permission.RECORD_AUDIO")
    AudioSource(@NonNull AudioSettings settings, @NonNull Executor executor, @Nullable Context attributionContext, @NonNull AudioStreamFactory audioStreamFactory, long startRetryIntervalMs) throws AudioSourceAccessException {
        this.mExecutor = CameraXExecutors.newSequentialExecutor((Executor)executor);
        this.mStartRetryIntervalNs = TimeUnit.MILLISECONDS.toNanos(startRetryIntervalMs);
        try {
            this.mAudioStream = new BufferedAudioStream(audioStreamFactory.create(settings, attributionContext), settings);
        }
        catch (AudioStream.AudioStreamException | IllegalArgumentException e) {
            throw new AudioSourceAccessException("Unable to create AudioStream", e);
        }
        this.mAudioStream.setCallback(new AudioStreamCallback(), this.mExecutor);
        this.mSilentAudioStream = new SilentAudioStream(settings);
        this.mAudioFormat = settings.getAudioFormat();
        this.mAudioSource = settings.getAudioSource();
    }

    public void setBufferProvider(@NonNull BufferProvider<? extends InputBuffer> bufferProvider) {
        this.mExecutor.execute(() -> {
            switch (this.mState.ordinal()) {
                case 0: 
                case 1: {
                    if (this.mBufferProvider == bufferProvider) break;
                    this.resetBufferProvider(bufferProvider);
                    break;
                }
                case 2: {
                    throw new AssertionError((Object)"AudioSource is released");
                }
            }
        });
    }

    public void start() {
        this.mExecutor.execute(() -> this.start(this.mMuted));
    }

    public void start(boolean muted) {
        this.mExecutor.execute(() -> {
            switch (this.mState.ordinal()) {
                case 0: {
                    this.mNotifiedSilenceState.set(null);
                    this.mNotifiedSuspendState.set(false);
                    this.setState(InternalState.STARTED);
                    this.mute(muted);
                    this.updateSendingAudio();
                    break;
                }
                case 1: {
                    break;
                }
                case 2: {
                    throw new AssertionError((Object)"AudioSource is released");
                }
            }
        });
    }

    public void stop() {
        this.mExecutor.execute(() -> {
            switch (this.mState.ordinal()) {
                case 1: {
                    this.setState(InternalState.CONFIGURED);
                    this.updateSendingAudio();
                    break;
                }
                case 0: {
                    break;
                }
                case 2: {
                    Logger.w((String)TAG, (String)"AudioSource is released. Calling stop() is a no-op.");
                }
            }
        });
    }

    public @NonNull ListenableFuture<Void> release() {
        return CallbackToFutureAdapter.getFuture(completer -> {
            this.mExecutor.execute(() -> {
                try {
                    switch (this.mState.ordinal()) {
                        case 0: 
                        case 1: {
                            this.resetBufferProvider(null);
                            this.mSilentAudioStream.release();
                            this.mAudioStream.release();
                            this.stopSendingAudio();
                            this.setState(InternalState.RELEASED);
                            break;
                        }
                    }
                    completer.set(null);
                }
                catch (Throwable t) {
                    completer.setException(t);
                }
            });
            return "AudioSource-release";
        });
    }

    public void setAudioSourceCallback(@NonNull Executor executor, @NonNull AudioSourceCallback callback) {
        this.mExecutor.execute(() -> {
            switch (this.mState.ordinal()) {
                case 0: {
                    this.mCallbackExecutor = executor;
                    this.mAudioSourceCallback = callback;
                    break;
                }
                case 1: 
                case 2: {
                    throw new AssertionError((Object)"The audio recording callback must be registered before the audio source is started.");
                }
            }
        });
    }

    public void mute(boolean muted) {
        this.mExecutor.execute(() -> {
            switch (this.mState.ordinal()) {
                case 0: 
                case 1: {
                    if (this.mMuted == muted) {
                        return;
                    }
                    this.mMuted = muted;
                    if (this.mState != InternalState.STARTED) break;
                    this.notifySilenced();
                    break;
                }
                case 2: {
                    throw new AssertionError((Object)"AudioSource is released");
                }
            }
        });
    }

    private void resetBufferProvider(final @Nullable BufferProvider<? extends InputBuffer> bufferProvider) {
        if (this.mBufferProvider != null) {
            this.mBufferProvider.removeObserver(Objects.requireNonNull(this.mStateObserver));
            this.mBufferProvider = null;
            this.mStateObserver = null;
            this.mAcquireBufferCallback = null;
            this.mBufferProviderState = BufferProvider.State.INACTIVE;
            this.updateSendingAudio();
        }
        if (bufferProvider != null) {
            this.mBufferProvider = bufferProvider;
            this.mStateObserver = new Observable.Observer<BufferProvider.State>(){

                public void onNewData(@Nullable BufferProvider.State state) {
                    Objects.requireNonNull(state);
                    if (AudioSource.this.mBufferProvider == bufferProvider) {
                        Logger.d((String)AudioSource.TAG, (String)("Receive BufferProvider state change: " + (Object)((Object)AudioSource.this.mBufferProviderState) + " to " + (Object)((Object)state)));
                        if (AudioSource.this.mBufferProviderState != state) {
                            AudioSource.this.mBufferProviderState = state;
                            AudioSource.this.updateSendingAudio();
                        }
                    }
                }

                public void onError(@NonNull Throwable throwable) {
                    if (AudioSource.this.mBufferProvider == bufferProvider) {
                        AudioSource.this.notifyError(throwable);
                    }
                }
            };
            this.mAcquireBufferCallback = new FutureCallback<InputBuffer>(){

                public void onSuccess(InputBuffer inputBuffer) {
                    ByteBuffer byteBuffer;
                    AudioStream audioStream;
                    AudioStream.PacketInfo packetInfo;
                    if (!AudioSource.this.mIsSendingAudio || AudioSource.this.mBufferProvider != bufferProvider) {
                        inputBuffer.cancel();
                        return;
                    }
                    if (AudioSource.this.mInSilentStartState && AudioSource.this.isStartRetryIntervalReached()) {
                        AudioSource.this.retryStartAudioStream();
                    }
                    if ((packetInfo = (audioStream = AudioSource.this.getCurrentAudioStream()).read(byteBuffer = inputBuffer.getByteBuffer())).getSizeInBytes() > 0) {
                        if (AudioSource.this.mMuted) {
                            AudioSource.this.overrideBySilence(byteBuffer, packetInfo.getSizeInBytes());
                        }
                        if (AudioSource.this.mCallbackExecutor != null && packetInfo.getTimestampNs() - AudioSource.this.mAmplitudeTimestamp >= 200L) {
                            AudioSource.this.mAmplitudeTimestamp = packetInfo.getTimestampNs();
                            AudioSource.this.postMaxAmplitude(byteBuffer);
                        }
                        byteBuffer.limit(byteBuffer.position() + packetInfo.getSizeInBytes());
                        inputBuffer.setPresentationTimeUs(TimeUnit.NANOSECONDS.toMicros(packetInfo.getTimestampNs()));
                        inputBuffer.submit();
                    } else {
                        Logger.w((String)AudioSource.TAG, (String)"Unable to read data from AudioStream.");
                        inputBuffer.cancel();
                    }
                    AudioSource.this.sendNextAudio();
                }

                public void onFailure(@NonNull Throwable throwable) {
                    if (AudioSource.this.mBufferProvider != bufferProvider) {
                        return;
                    }
                    Logger.d((String)AudioSource.TAG, (String)"Unable to get input buffer, the BufferProvider could be transitioning to INACTIVE state.");
                    if (!(throwable instanceof IllegalStateException)) {
                        AudioSource.this.notifyError(throwable);
                    }
                }
            };
            BufferProvider.State state = AudioSource.fetchBufferProviderState(bufferProvider);
            if (state != null) {
                this.mBufferProviderState = state;
                this.updateSendingAudio();
            }
            this.mBufferProvider.addObserver(this.mExecutor, this.mStateObserver);
        }
    }

    @NonNull AudioStream getCurrentAudioStream() {
        return this.mInSilentStartState ? this.mSilentAudioStream : this.mAudioStream;
    }

    void retryStartAudioStream() {
        Preconditions.checkState((boolean)this.mInSilentStartState);
        try {
            this.mAudioStream.start();
            Logger.d((String)TAG, (String)"Retry start AudioStream succeed");
            this.mSilentAudioStream.stop();
            this.mInSilentStartState = false;
        }
        catch (AudioStream.AudioStreamException e) {
            Logger.w((String)TAG, (String)"Retry start AudioStream failed", (Throwable)e);
            this.mLatestFailedStartTimeNs = AudioSource.getCurrentSystemTimeNs();
        }
    }

    boolean isStartRetryIntervalReached() {
        Preconditions.checkState((this.mLatestFailedStartTimeNs > 0L ? 1 : 0) != 0);
        return AudioSource.getCurrentSystemTimeNs() - this.mLatestFailedStartTimeNs >= this.mStartRetryIntervalNs;
    }

    void notifyError(@NonNull Throwable throwable) {
        Executor executor = this.mCallbackExecutor;
        AudioSourceCallback callback = this.mAudioSourceCallback;
        if (executor != null && callback != null) {
            executor.execute(() -> callback.onError(throwable));
        }
    }

    void notifySilenced() {
        Executor executor = this.mCallbackExecutor;
        AudioSourceCallback callback = this.mAudioSourceCallback;
        if (executor != null && callback != null) {
            boolean isSilenced;
            boolean bl = isSilenced = this.mMuted || this.mInSilentStartState || this.mAudioStreamSilenced;
            if (!Objects.equals(this.mNotifiedSilenceState.getAndSet(isSilenced), isSilenced)) {
                executor.execute(() -> callback.onSilenceStateChanged(isSilenced));
            }
        }
    }

    void notifySuspended(boolean isSuspended) {
        Executor executor = this.mCallbackExecutor;
        AudioSourceCallback callback = this.mAudioSourceCallback;
        if (executor != null && callback != null && this.mNotifiedSuspendState.getAndSet(isSuspended) != isSuspended) {
            executor.execute(() -> callback.onSuspendStateChanged(isSuspended));
        }
    }

    void overrideBySilence(@NonNull ByteBuffer byteBuffer, int sizeInBytes) {
        if (this.mZeroBytes == null || this.mZeroBytes.length < sizeInBytes) {
            this.mZeroBytes = new byte[sizeInBytes];
        }
        int positionBeforePut = byteBuffer.position();
        byteBuffer.put(this.mZeroBytes, 0, sizeInBytes);
        byteBuffer.limit(byteBuffer.position()).position(positionBeforePut);
    }

    void updateSendingAudio() {
        if (this.mState == InternalState.STARTED) {
            boolean isBufferProviderActive = this.mBufferProviderState == BufferProvider.State.ACTIVE;
            this.notifySuspended(!isBufferProviderActive);
            if (isBufferProviderActive) {
                this.startSendingAudio();
            } else {
                this.stopSendingAudio();
            }
        } else {
            this.stopSendingAudio();
        }
    }

    private void startSendingAudio() {
        if (this.mIsSendingAudio) {
            return;
        }
        try {
            Logger.d((String)TAG, (String)"startSendingAudio");
            this.mAudioStream.start();
            this.mInSilentStartState = false;
        }
        catch (AudioStream.AudioStreamException e) {
            Logger.w((String)TAG, (String)"Failed to start AudioStream", (Throwable)e);
            this.mInSilentStartState = true;
            this.mSilentAudioStream.start();
            this.mLatestFailedStartTimeNs = AudioSource.getCurrentSystemTimeNs();
            this.notifySilenced();
        }
        this.mIsSendingAudio = true;
        this.sendNextAudio();
    }

    private void stopSendingAudio() {
        if (!this.mIsSendingAudio) {
            return;
        }
        this.mIsSendingAudio = false;
        Logger.d((String)TAG, (String)"stopSendingAudio");
        this.mAudioStream.stop();
    }

    void sendNextAudio() {
        Futures.addCallback(Objects.requireNonNull(this.mBufferProvider).acquireBuffer(), Objects.requireNonNull(this.mAcquireBufferCallback), (Executor)this.mExecutor);
    }

    void setState(InternalState state) {
        Logger.d((String)TAG, (String)("Transitioning internal state: " + (Object)((Object)this.mState) + " --> " + (Object)((Object)state)));
        this.mState = state;
    }

    void postMaxAmplitude(ByteBuffer byteBuffer) {
        Executor executor = this.mCallbackExecutor;
        AudioSourceCallback callback = this.mAudioSourceCallback;
        double maxAmplitude = 0.0;
        if (this.mAudioFormat == 2) {
            ShortBuffer shortBuffer = byteBuffer.asShortBuffer();
            while (shortBuffer.hasRemaining()) {
                maxAmplitude = Math.max(maxAmplitude, (double)Math.abs(shortBuffer.get()));
            }
            this.mAudioAmplitude = maxAmplitude /= 32767.0;
            if (executor != null && callback != null) {
                executor.execute(() -> callback.onAmplitudeValue(this.mAudioAmplitude));
            }
        }
    }

    private static @Nullable BufferProvider.State fetchBufferProviderState(@NonNull BufferProvider<? extends InputBuffer> bufferProvider) {
        try {
            ListenableFuture state = bufferProvider.fetchData();
            return state.isDone() ? (BufferProvider.State)((Object)state.get()) : null;
        }
        catch (InterruptedException | ExecutionException e) {
            return null;
        }
    }

    public static boolean isSettingsSupported(int sampleRate, int channelCount, int audioFormat) {
        return AudioStreamImpl.isSettingsSupported(sampleRate, channelCount, audioFormat);
    }

    private static long getCurrentSystemTimeNs() {
        return System.nanoTime();
    }

    static enum InternalState {
        CONFIGURED,
        STARTED,
        RELEASED;

    }

    class AudioStreamCallback
    implements AudioStream.AudioStreamCallback {
        AudioStreamCallback() {
        }

        @Override
        public void onSilenceStateChanged(boolean isSilenced) {
            AudioSource.this.mAudioStreamSilenced = isSilenced;
            if (AudioSource.this.mState == InternalState.STARTED) {
                AudioSource.this.notifySilenced();
            }
        }
    }

    public static interface AudioSourceCallback {
        @VisibleForTesting
        default public void onSuspendStateChanged(boolean suspended) {
        }

        public void onSilenceStateChanged(boolean var1);

        public void onError(@NonNull Throwable var1);

        public void onAmplitudeValue(double var1);
    }
}

